/**
 * Copyright IBM Corp. 2015, 2025
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

'use strict';

import 'core-js/features/array/flat-map.js';
import { reporter } from '@carbon/cli-reporter';
import { types as t, generate } from '@carbon/scss-generator';
import fs from 'fs-extra';
import path from 'path';
import { fileURLToPath } from 'url';
import {
  container,
  iconSize,
  spacing,
  fluidSpacing,
  sizes,
  layout,
} from '../src/index.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

async function build() {
  reporter.info('Building scss files for layout...');

  const SCSS_DIR = path.resolve(__dirname, '../scss/generated');
  const files = [
    {
      filepath: path.join(SCSS_DIR, '_container.scss'),
      builder() {
        return buildModulesTokenFile(container, 'container');
      },
    },
    {
      filepath: path.join(SCSS_DIR, '_size.scss'),
      builder() {
        const FILE_BANNER =
          t.Comment(` Code generated by @carbon/layout. DO NOT EDIT.

 Copyright IBM Corp. 2018, 2023

 This source code is licensed under the Apache-2.0 license found in the
 LICENSE file in the root directory of this source tree.
`);
        const convert = {
          XSmall: 'xs',
          Small: 'sm',
          Medium: 'md',
          Large: 'lg',
          XLarge: 'xl',
          '2XLarge': '2xl',
        };

        const comment = t.Comment(`/ @type Number
/ @access public
/ @group @carbon/layout`);

        return t.StyleSheet([
          FILE_BANNER,
          t.Newline(),
          comment,
          ...Object.entries(sizes).map(([name, value]) => {
            return t.Assignment({
              id: t.Identifier(`size-${convert[name]}`),
              init: t.SassValue(value),
            });
          }),
        ]);
      },
    },
    {
      filepath: path.join(SCSS_DIR, '_icon-size.scss'),
      builder() {
        return buildModulesTokenFile(iconSize, 'icon-size');
      },
    },
    {
      filepath: path.join(SCSS_DIR, '_spacing.scss'),
      builder() {
        return buildModulesTokenFile(spacing, 'spacing', '');
      },
    },
    {
      filepath: path.join(SCSS_DIR, '_fluid-spacing.scss'),
      builder() {
        return buildModulesTokenFile(fluidSpacing, 'fluid-spacing', '');
      },
    },
    {
      filepath: path.join(SCSS_DIR, '_layout.scss'),
      builder() {
        return buildModulesTokenFile(layout, 'layout');
      },
    },
  ];

  await fs.ensureDir(SCSS_DIR);
  for (const { filepath, builder } of files) {
    const { code } = generate(builder());
    await fs.writeFile(filepath, await code);
  }

  reporter.success('Done! 🎉');
}

/**
 * Build a Sass Module token stylesheet for a given token scale and group. This will help
 * generate the initial collection of tokens and a list of all tokens. In
 * addition, it will generate aliases for these tokens.
 *
 * @param {Array} tokenScale
 * @param {string} group
 * @returns {StyleSheet}
 */
function buildModulesTokenFile(tokenScale, group) {
  const FILE_BANNER = t.Comment(` Code generated by @carbon/layout. DO NOT EDIT.

 Copyright IBM Corp. 2018, 2023

 This source code is licensed under the Apache-2.0 license found in the
 LICENSE file in the root directory of this source tree.
`);

  const values = tokenScale.map((value, index) => {
    const name = formatStep(`${group}`, index + 1);
    const shorthand = formatStep(group, index + 1);
    const id = t.Identifier(name);
    return [
      name,
      shorthand,
      id,
      t.Assignment({
        id,
        init: t.SassValue(value),
        default: true,
      }),
    ];
  });

  const variables = values.flatMap(([, , , assignment]) => {
    const comment = t.Comment(`/ @type Number
/ @access public
/ @group @carbon/layout`);
    return [comment, assignment, t.Newline()];
  });

  const map = [
    t.Comment(`/ @type Map
/ @access public
/ @group @carbon/layout`),
    t.Assignment({
      id: t.Identifier(group),
      init: t.SassMap({
        properties: values.map(([name, , id]) => {
          return t.SassMapProperty({
            key: id,
            value: t.SassValue(`$${name}`),
          });
        }),
      }),
    }),
  ];

  return t.StyleSheet([
    FILE_BANNER,
    t.Newline(),
    ...variables,
    ...map,
    t.Newline(),
  ]);
}

/**
 * Format the given step for a token name. Most often, this is to pad a `0` for
 * numbers that are less than 10. For example, instead of spacing-1 we would
 * want spacing-01
 *
 * @param {string} name
 * @param {number} index
 * @returns {string}
 */
function formatStep(name, index) {
  let step = index;
  if (step < 10) {
    step = '0' + step;
  }
  return `${name}-${step}`;
}

build().catch((error) => {
  console.log(error);
  process.exit(1);
});
